import nodemailer from 'nodemailer';

export const config = { api: { bodyParser: true } };

function id() {
  const d = new Date().toISOString().slice(0,10).replace(/-/g,'');
  const r = Math.floor(Math.random()*8999+1000);
  return `DSAR-${d}-${r}`;
}

export default async function handler(req, res) {
  if (req.method !== 'POST') return res.status(405).json({ error: 'Method not allowed' });
  const { name, email, country, type, details, consent } = req.body;
  if (!name || !email || !country || !type || consent !== 'on') {
    return res.status(400).json({ error: 'Missing required fields or consent' });
  }

  const requestId = id();
  const ackHtml = `<p>Dear ${name},</p>
  <p>We have received your data subject request. Your reference ID is <b>${requestId}</b>.</p>
  <p>We will respond within 30 days in accordance with GDPR.</p>
  <p>Regards,<br/>Xyberteq Privacy Team</p>`;

  try {
    // Email transport (configure in .env)
    const transporter = nodemailer.createTransport({
      host: process.env.SMTP_HOST,
      port: Number(process.env.SMTP_PORT || 587),
      secure: false,
      auth: { user: process.env.SMTP_USER, pass: process.env.SMTP_PASS },
    });

    // Acknowledge to data subject
    await transporter.sendMail({
      from: process.env.PRIVACY_FROM || 'privacy@xyberteq.com',
      to: email,
      subject: `Your DSAR request: ${requestId}`,
      html: ackHtml,
    });

    // Notify internal privacy team
    const body = `New DSAR:
RequestID: ${requestId}
Name: ${name}
Email: ${email}
Country: ${country}
Type: ${type}
Details: ${details}`;

    await transporter.sendMail({
      from: process.env.PRIVACY_FROM || 'privacy@xyberteq.com',
      to: process.env.PRIVACY_TEAM || 'privacy-team@xyberteq.com',
      subject: `NEW DSAR: ${requestId}`,
      text: body,
    });

    // TODO: Persist to your DB or Google Sheets via API for tracking

    return res.status(200).json({ ok: true, requestId });
  } catch (e) {
    console.error(e);
    return res.status(500).json({ error: 'Failed to send emails' });
  }
}
